/**
 * @file CGGetRankingAtIndex.js
 * @author Frank Hamilton
 * @date 2024/2/27
 * @brief CGGetRankingAtIndex.js
 * @copyright Copyright (c) 2023, ByteDance Inc, All Rights Reserved
 */

const {BaseNode} = require('../Utils/BaseNode');
const Forge = require('../../../forge.min');
const {LeaderboardDataStore} = require('../../../Plugins/LeaderboardDataStore');

const Amaz = effect.Amaz;
const LBSDS = LeaderboardDataStore.instance;

class CGGetRankingAtIndex extends BaseNode {
  constructor() {
    super();
  }

  onInit() {
    this.initialSettings();
    this.outputs[1] = false;
    this.outputs[2] = '';
    this.outputs[3] = this.releatedRes.get('00000000-0000-0000-0000-000000004007');
    this.outputs[4] = -1;
    this.outputs[5] = false;
    this.outputs[6] = -1;
  }

  resetOnRecord(sys) {
    this.initialSettings();
  }

  initialSettings() {
    this.triggerPulse = false;
    if (this.releatedRes) {
      LBSDS.defaultIcon = this.releatedRes.get('00000000-0000-0000-0000-000000004007');
    }
  }

  setNext(index, func) {
    this.nexts[index] = func;
  }

  setInput(index, func) {
    this.inputs[index] = func;
  }

  execute(index) {
    if (index === 0) {
      if (!LBSDS) return;
      if (LBSDS && LBSDS.leaderboardReady === true) {
        this.processLeaderboardInformation();
      } else {
        this.badCall();
      }
    }
  }

  processLeaderboardInformation() {
    const index = Math.floor(this.inputs[1]() + 0.5);
    if (index > 6 || index < 1) {
      this.badCall();
      this.triggerPulse = true;
      return;
    }
    const leaderboardType = this.inputs[2]();

    if (LBSDS.scoreSet === true) {
      this.process_scoreSet(leaderboardType, index);
    } else {
      this.process_scoreNotSet(leaderboardType, index);
    }
  }

  process_scoreSet(leaderboardType, index) {
    let leaderboard;
    let currentRank;
    let bestRank;

    switch (leaderboardType) {
      case 'Friend':
        leaderboard = LBSDS.friends_leaderboard_minusBest;
        currentRank = LBSDS.currentRank_friends;
        if (LBSDS.hasScore && LBSDS.currentRank_friends > LBSDS.friends_rank) {
          currentRank = currentRank - 1;
        }
        bestRank = LBSDS.friends_rank;
        break;
      case 'National':
        leaderboard = LBSDS.national_leaderboard_minusBest;
        currentRank = LBSDS.currentRank_national;
        if (LBSDS.hasScore && LBSDS.currentRank_national > LBSDS.national_rank) {
          currentRank = currentRank - 1;
        }
        bestRank = LBSDS.national_rank;
        break;
      case 'Global':
        leaderboard = LBSDS.global_leaderboard_minusBest;
        currentRank = LBSDS.currentRank_global;
        if (LBSDS.hasScore && LBSDS.currentRank_global > LBSDS.global_rank) {
          currentRank = currentRank - 1;
        }
        bestRank = LBSDS.global_rank;
        break;
    }

    if ((index >= 6 && currentRank >= 6) || index === currentRank) {
      this.outputs[1] = true;
      this.outputs[2] = LBSDS.userProfile_Data.nickname;
      LBSDS.getImage(LBSDS.userProfile_Data.avatar_path, this.imageCallback.bind(this));
      this.outputs[4] = LBSDS.currentScore;
      this.outputs[5] = true;
      this.outputs[6] = currentRank;
      return;
    }
    if (index >= currentRank) {
      index = index - 1;
    }

    const boardEntry = leaderboard[index - 1];
    let ranking = LBSDS.getOtherRank(boardEntry.score, leaderboardType);
    if (ranking > bestRank && bestRank > 0) {
      ranking = ranking - 1;
    }
    if (ranking >= currentRank && boardEntry.score !== LBSDS.currentScore) {
      ranking = ranking + 1;
    }

    if (boardEntry) {
      this.outputs[1] = true;
      this.outputs[2] = boardEntry.username;
      const ius = boardEntry.icon_url.url_list;
      const url = LBSDS.getBestURL(ius, false);
      if (url === null) {
        if (LBSDS.defaultIcon) {
          this.outputs[3] = LBSDS.defaultIcon;
        }
        this.triggerPulse = true;
      } else {
        LBSDS.getImage(url, this.imageCallback.bind(this));
      }

      this.outputs[4] = boardEntry.score;
      this.outputs[5] = boardEntry.uid_str === LBSDS.userInfo_Data.userID;
      this.outputs[6] = ranking;
    } else {
      this.badCall();
      this.triggerPulse = true;
    }
  }

  process_scoreNotSet(leaderboardType, index) {
    let leaderboard;
    let bestRank;

    switch (leaderboardType) {
      case 'Friend':
        leaderboard = LBSDS.friends_leaderboard.leaderboard;
        bestRank = LBSDS.friends_rank;
        break;
      case 'National':
        leaderboard = LBSDS.national_leaderboard.leaderboard;
        bestRank = LBSDS.national_rank;
        break;
      case 'Global':
        leaderboard = LBSDS.global_leaderboard.leaderboard;
        bestRank = LBSDS.global_rank;
        break;
    }

    if (bestRank >= 6 && index === 6) {
      this.outputs[1] = true;
      this.outputs[2] = LBSDS.userProfile_Data.nickname;
      LBSDS.getImage(LBSDS.userProfile_Data.avatar_path, this.imageCallback.bind(this));
      this.outputs[4] = LBSDS.bestScore;
      this.outputs[5] = true;
      this.outputs[6] = bestRank;

      return;
    }

    const boardEntry = leaderboard[index - 1];

    const ranking = LBSDS.getOtherRank(boardEntry.score, leaderboardType);

    if (boardEntry) {
      this.outputs[1] = true;
      this.outputs[2] = boardEntry.username;
      const ius = boardEntry.icon_url.url_list;
      const url = LBSDS.getBestURL(ius, false);
      if (url === null) {
        if (LBSDS.defaultIcon) {
          this.outputs[3] = LBSDS.defaultIcon;
        }
        this.triggerPulse = true;
      } else {
        LBSDS.getImage(url, this.imageCallback.bind(this));
      }

      this.outputs[4] = boardEntry.score;
      this.outputs[5] = boardEntry.uid_str === LBSDS.userInfo_Data.userID;
      this.outputs[6] = ranking;
    } else {
      this.badCall();
      this.triggerPulse = true;
    }
  }

  imageCallback(image) {
    if (!image) return;

    this.outputs[3] = image;
    this.triggerPulse = true;
  }

  onUpdate(sys, dt) {
    if (!LBSDS) return;

    if (LBSDS.leaderboardReady === true && this.triggerPulse === true) {
      this.triggerPulse = false;
      if (this.nexts[0]) {
        this.nexts[0]();
      }
    }
  }

  badCall() {
    this.outputs[1] = false;
    this.outputs[2] = '';
    if (LBSDS.defaultIcon) {
      this.outputs[3] = LBSDS.defaultIcon;
    }
    this.outputs[4] = -1;
    this.outputs[5] = false;
  }
}

exports.CGGetRankingAtIndex = CGGetRankingAtIndex;
